// Handles updating elements with live lamport values and converting to fiat if user has a preferred fiat set
let SOLANA_LIVE_RATES, SOLANA_RATES_TTL, SOLANA_RATES_URL, SOLANA_RATES_USE_FIAT = null;

function fiat_symbol(arg = null) {
    switch (arg) {
        case 'USD': return '$';
        case 'AUD': return 'A$';
        case 'EUR': return '€';
        case 'GBP': return '£';
        case 'CAD': return 'C$';
        default: ;
    }
    return '[?]';
}

function _get_rate() {
    if (!SOLANA_LIVE_RATES || typeof SOLANA_LIVE_RATES !== 'object' || !SOLANA_RATES_USE_FIAT) {
        console.error('Invalid SOLANA_LIVE_RATES:');
        return null;
    }
    // Get rate for user's preferred fiat
    if (SOLANA_LIVE_RATES[SOLANA_RATES_USE_FIAT] && !isNaN(SOLANA_LIVE_RATES[SOLANA_RATES_USE_FIAT])) {
        return SOLANA_LIVE_RATES[SOLANA_RATES_USE_FIAT] * 1;
    }
    // Check if rate is available for SOL[fiat]
    if (SOLANA_LIVE_RATES[`SOL${SOLANA_RATES_USE_FIAT}`] && !isNaN(SOLANA_LIVE_RATES[`SOL${SOLANA_RATES_USE_FIAT}`])) {
        return SOLANA_LIVE_RATES[`SOL${SOLANA_RATES_USE_FIAT}`] * 1;
    }
    // Check if rate is available for SOL_[fiat]
    if (SOLANA_LIVE_RATES[`SOL_${SOLANA_RATES_USE_FIAT}`] && !isNaN(SOLANA_LIVE_RATES[`SOL_${SOLANA_RATES_USE_FIAT}`])) {
        return SOLANA_LIVE_RATES[`SOL_${SOLANA_RATES_USE_FIAT}`] * 1;
    }
    return null;
}

function update_lamport_elements() {
    // Check if SOLANA_LIVE_RATES is defined and if user's preferred fiat is available

    $('.live_lamports').each(function() {
        try {
            // Get lamports from data-lamports attribute and convert to sol
            const lamports = parseInt($(this).attr('data-lamports'));
            const sol = lamports / 1000000000;
            const sol_rate = _get_rate();
            const sol_str = `${sol} SOL`;
            if (!sol_rate) {
                // Create a value string without trailing zeros
                $(this).empty().append(`<span style="font-weight:600;">${sol_str}</span>`);
                return;
            }
            // Convert to fiat
            const fiat = sol * sol_rate;
            const fiat_locale_fixed = fiat.toLocaleString(undefined, { minimumFractionDigits: 2, maximumFractionDigits: 2 });
            // Update element
            $(this).empty().append(`<span style="font-weight:600;">${fiat_symbol(SOLANA_RATES_USE_FIAT)}${fiat_locale_fixed}</span>`);
            if ($(this).hasClass('show_sol')) {
                $(this).append(`&nbsp;&nbsp;<span style="opacity:0.3;font-weight:300;">|</span>&nbsp;&nbsp;<span style="font-weight:300;">${sol_str}</span>`);
            }
        } catch (e) {
            console.error('Error updating live lamport element:', e);
        }
    });
}

function update_solana_rates() {
    fetch(SOLANA_RATES_URL)
        .then(response => response.json())
        .then(data => {
            try {
                SOLANA_LIVE_RATES = data?.rates || {};
                SOLANA_RATES_TTL = data?.ttl || 120; // seconds
                SOLANA_RATES_TTL = SOLANA_RATES_TTL * 1000; // convert to milliseconds
                update_lamport_elements();
                setTimeout(() => {
                    update_solana_rates();
                }, SOLANA_RATES_TTL);
            } catch (e) {
                console.error('Error updating SOLANA_LIVE_RATES:', e);
            }
        });
}

function init_live_lamports(rates_url = '/static/rates.json') {
    SOLANA_RATES_URL = rates_url;
    const user_fiat = $('#user_preferred_fiat').val();
    if (user_fiat && typeof user_fiat === 'string' && user_fiat.trim().length > 0) {
        SOLANA_RATES_USE_FIAT = user_fiat.trim();
    }
    update_solana_rates();
}